#include <stdio.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <vector>
#include <list>
#include <signal.h>

// IR Imager device interfaces
#include "IRDeviceDS.h"

// Raw video interface
#include "VideoCapture.h"

// IR Imager imager interfaces
#include "IRImager.h"

// IR Imager logging interface
#include "IRLogger.h"

// IR Imager raw image file writer
#include "IRFileWriter.h"

// Time measurement class
#include "Timer.h"

using namespace std;
using namespace evo;

IRImager*     _imager = NULL;
IRFileWriter* _writer = NULL;
char          _nmea[GPSBUFFERSIZE];
Timer         _timer;

bool          _keepCapturing = true;
int           _serializedImages = 0;
int           _chunk = 1;

// Function called be DirectShow interface when a new frame has been acquired.
void onRawFrame(unsigned char* data, int len, int id)
{
  if (!_keepCapturing) return;

  // The method process needs to be called in order to give control to the library, e.g. to control the shutter flag
  _imager->process(data, NULL);

  if(_writer)
  {
    double timestamp = _timer.getTime();
    if (_writer->canDoWriteOperations())
    {
      _writer->write(timestamp, data, _chunk, len, _nmea);
      cout << ".";
    }
  }

  // In order to avoid too large files, the user can split the records into chunks of equal size.
  // Here, a fixed number of images should be recorded to each chunk.
  if ((++_serializedImages) % 100 == 0)
  {
    _chunk++;
    if (_chunk < 4)
      cout << endl << "Opening next chunk (" << _chunk << ")" << endl;
    else
      _keepCapturing = false;
  }
}

int main (int argc, char* argv[])
{
  if (argc != 2)
  {
    cout << "usage: " << argv[0] << " <xml configuration file>" << endl;
    return -1;
  }

  char* args = argv[1];
  // Windows SDK is compiled using 16-bit Unicode characters
  size_t argSize = strlen(args) + 1, converted;
  wchar_t* argPath = new wchar_t[argSize];
  mbstowcs_s(&converted, argPath, argSize, args, argSize);

  IRDeviceParams p;
  if (IRDeviceParamsReader::readXML(argPath, p) == false)
    return -1;
  delete argPath;

  VideoCapture vc;
  IRDeviceDS* device = vc.initializeDevice(p);

  if(device)
  {

    device->setCallback(onRawFrame);

    _imager = new IRImager();

    if(_imager->init(&p, device->getFrequency(), device->getWidth(), device->getHeight()))
    {

      RawdataHeader header;
      _imager->initRawdataHeader(header);

      // A file sequence is created having the following pattern "ir_<serial>_<date>_<time>.raw.x" and "ir_<serial>_<date>_<time>.time.x"
      // The *.raw files can be importex to PIConnect. The *.time files contain the raw frame's time stamps.
      _writer = new IRFileWriter(time(NULL), ".", header);
      if(_writer->open())
      {
        vc.run();

        memset(_nmea, 0, GPSBUFFERSIZE*sizeof(*_nmea));

        // A flag event must take place at the beginning of the record process
        _imager->forceFlagEvent(1000.0);

        device->startStreaming();

        // The main thread has nothing to do in this example
        while (_keepCapturing)
        {
          Sleep(1);
        }

        device->stopStreaming();
      }
    }
  }
  else
  {
    std::cout << "IR Imager device(s) could not be found" << std::endl;
  }

  if(_imager) delete _imager;

  if(_writer) delete _writer;

  std::cout << endl << "Exiting application ... serialized " << _serializedImages << " images" << std::endl;

  return 0;
}
